/* Copyright (C) 2015-2018 RealVNC Ltd.  All Rights Reserved.
 */

#include <vnccommon/Clock.h>

#include <sys/time.h>
#include <time.h>

using namespace vnccommon;

namespace
{

    /* Ensure that all arithmetic is done as unsigned so that any
     * overflow is defined behaviour.
     *
     * Overflows are acceptable as the time returned by this
     * function is only used as a delta.
     */
#if !defined(__APPLE__)
    /*
     * MOB-14963: the following function is only used under Linux but
     * not used under Mac. When this file is compiled under Mac by Clang,
     * there will be an unused function warning if this function is not
     * removed by ifdef.
     */
    MonotonicTimestamp convertToTimestamp(const struct timespec& time)
    {
        return MonotonicTimestamp(static_cast<vnc_int64_t>(
                static_cast<vnc_uint64_t>(time.tv_sec) * 1000
                    + static_cast<vnc_uint64_t>(time.tv_nsec) / 1000000
        ));
    }
#endif

    MonotonicTimestamp convertToTimestamp(const struct timeval& time)
    {
        return MonotonicTimestamp(static_cast<vnc_int64_t>(
                static_cast<vnc_uint64_t>(time.tv_sec) * 1000
                    + static_cast<vnc_uint64_t>(time.tv_usec) / 1000
        ));
    }

} // end of anonymous namespace

MonotonicTimestamp Clock::getCurrentTimeMonotonic()
{
    /* Mac OS X doesn't support clock_gettime so always fallback on
     * gettimeofday. */
#if !defined(__APPLE__)
    struct timespec time;

    if (clock_gettime(CLOCK_MONOTONIC, &time) == 0)
    {
        return convertToTimestamp(time);
    }
    else
#endif
    {
        /* Fallback to gettimeofday which has been used prior to the fix
         * for MOB-9428 without many ill effects.
         *
         * According to POSIX, gettimeofday can only fail with EFAULT
         * for invalid memory and EINVAL for an invalid timezone
         * parameter. As the timezone is always NULL this shouldn't be
         * an issue.
         */
        struct timeval tv;
        gettimeofday(&tv, NULL);
        return convertToTimestamp(tv);
    }
}

